<?php
require_once __DIR__ . '/../config/database.php';

class User {
    private $conn;
    private $table_name = "users";

    public $id;
    public $username;
    public $password;
    public $email;
    public $user_type;
    public $created_at;
    public $reset_token;
    public $reset_token_expiry;

    public function __construct() {
        $this->conn = getDBConnection();
    }

    public function create() {
        $query = "INSERT INTO " . $this->table_name . "
                (username, password, email, user_type, created_at)
                VALUES
                (:username, :password, :email, :user_type, :created_at)";

        $stmt = $this->conn->prepare($query);

        // Sanitize and hash password
        $this->username = htmlspecialchars(strip_tags($this->username));
        $this->email = htmlspecialchars(strip_tags($this->email));
        $this->password = password_hash($this->password, PASSWORD_DEFAULT);
        $this->created_at = date('Y-m-d H:i:s');

        // Bind values
        $stmt->bindParam(":username", $this->username);
        $stmt->bindParam(":password", $this->password);
        $stmt->bindParam(":email", $this->email);
        $stmt->bindParam(":user_type", $this->user_type);
        $stmt->bindParam(":created_at", $this->created_at);

        if($stmt->execute()) {
            return true;
        }
        return false;
    }

    public function login($username, $password) {
        $query = "SELECT id, username, password, user_type FROM " . $this->table_name . "
                WHERE username = :username LIMIT 1";

        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":username", $username);
        $stmt->execute();

        if($stmt->rowCount() > 0) {
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            if(password_verify($password, $row['password'])) {
                return $row;
            }
        }
        return false;
    }

    public function getAllUsers() {
        $query = "SELECT id, username, email, user_type, created_at FROM " . $this->table_name;
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        return $stmt;
    }

    public function deleteUser($id) {
        $query = "DELETE FROM " . $this->table_name . " WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":id", $id);
        return $stmt->execute();
    }

    public function requestPasswordReset($email) {
        $query = "SELECT id, username FROM " . $this->table_name . " WHERE email = :email LIMIT 1";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":email", $email);
        $stmt->execute();

        if($stmt->rowCount() > 0) {
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            
            // Generate reset token
            $token = bin2hex(random_bytes(32));
            $expiry = date('Y-m-d H:i:s', strtotime('+1 hour'));
            
            // Update user with reset token
            $updateQuery = "UPDATE " . $this->table_name . " 
                          SET reset_token = :token, reset_token_expiry = :expiry 
                          WHERE id = :id";
            $updateStmt = $this->conn->prepare($updateQuery);
            $updateStmt->bindParam(":token", $token);
            $updateStmt->bindParam(":expiry", $expiry);
            $updateStmt->bindParam(":id", $row['id']);
            
            if($updateStmt->execute()) {
                return [
                    'success' => true,
                    'token' => $token,
                    'username' => $row['username']
                ];
            }
        }
        return ['success' => false];
    }

    public function validateResetToken($token) {
        $query = "SELECT id FROM " . $this->table_name . " 
                 WHERE reset_token = :token 
                 AND reset_token_expiry > NOW() 
                 LIMIT 1";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":token", $token);
        $stmt->execute();
        
        return $stmt->rowCount() > 0;
    }

    public function resetPassword($token, $new_password) {
        $query = "UPDATE " . $this->table_name . " 
                 SET password = :password, 
                     reset_token = NULL, 
                     reset_token_expiry = NULL 
                 WHERE reset_token = :token 
                 AND reset_token_expiry > NOW()";
        
        $stmt = $this->conn->prepare($query);
        $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
        $stmt->bindParam(":password", $hashed_password);
        $stmt->bindParam(":token", $token);
        
        return $stmt->execute();
    }
}
?> 